<?php
require_once 'config/config.php';

$pdo = getDBConnection();

// Get all active channels (pinned first, then by name)
try {
    $channels = $pdo->query("SELECT * FROM tv_channels WHERE is_active = 1 ORDER BY is_pinned DESC, name ASC")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // If is_pinned column doesn't exist, fallback to name order
    $channels = $pdo->query("SELECT * FROM tv_channels WHERE is_active = 1 ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);
}

// Get categories
$categories = $pdo->query("SELECT DISTINCT category FROM tv_channels WHERE is_active = 1 AND category IS NOT NULL ORDER BY category ASC")->fetchAll(PDO::FETCH_COLUMN);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Live TV Streaming - Watch Free TV Online</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: #000000;
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .navbar {
            background: rgba(139, 0, 0, 0.9);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .navbar-brand {
            font-weight: bold;
            font-size: 1.5rem;
        }
        
        .hero-section {
            background: linear-gradient(135deg, rgba(0, 0, 0, 0.8) 0%, rgba(139, 0, 0, 0.9) 100%);
            color: white;
            padding: 1rem 0;
            text-align: center;
        }
        
        .channel-card {
            background: rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            padding: 0.75rem;
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.2);
            transition: all 0.3s ease;
            cursor: pointer;
            border: 1px solid rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            height: 100%;
        }
        
        .channel-card:hover {
            transform: translateX(5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.3);
            border-color: #DC143C;
            background: rgba(255, 255, 255, 0.15);
        }
        
        .channel-logo {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid #DC143C;
        }
        
        .channel-logo-small {
            flex-shrink: 0;
        }
        
        .pinned-channel {
            border-left: 4px solid #ffc107 !important;
            background: rgba(255, 193, 7, 0.1) !important;
        }
        
        .pinned-channel:hover {
            background: rgba(255, 193, 7, 0.2) !important;
            border-left-color: #ffc107 !important;
        }
        
        .channel-name {
            font-weight: bold;
            color: white;
            margin-bottom: 0.25rem;
            font-size: 0.9rem;
        }
        
        .channel-category {
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            color: white;
            padding: 0.15rem 0.5rem;
            border-radius: 15px;
            font-size: 0.7rem;
            display: inline-block;
        }
        
        .video-player {
            background: #000;
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
        }
        
        .video-player video {
            width: 100%;
            height: auto;
            min-height: 500px;
        }
        
        .video-player-container {
            position: sticky;
            top: 0;
            z-index: 1000;
        }
        
        .welcome-message {
            background: rgba(255, 255, 255, 0.1);
            border-radius: 15px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .category-filter {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border-radius: 10px;
            padding: 0.75rem;
            margin-bottom: 1.5rem;
        }
        
        .filter-btn {
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            margin: 0.15rem;
            transition: all 0.3s;
            font-size: 0.85rem;
        }
        
        .filter-btn:hover, .filter-btn.active {
            background: rgba(255, 255, 255, 0.3);
            border-color: rgba(255, 255, 255, 0.5);
            transform: translateY(-2px);
        }
        
        .search-box {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 25px;
            padding: 0.75rem 1.5rem;
            color: white;
        }
        
        .search-box::placeholder {
            color: rgba(255, 255, 255, 0.7);
        }
        
        .search-box:focus {
            background: rgba(255, 255, 255, 0.2);
            border-color: rgba(255, 255, 255, 0.5);
            box-shadow: 0 0 0 0.2rem rgba(255, 255, 255, 0.25);
            color: white;
        }
        
        .no-channels {
            text-align: center;
            padding: 4rem 2rem;
            color: rgba(255, 255, 255, 0.8);
        }
        
        .admin-link {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            color: white;
            padding: 1rem;
            border-radius: 50%;
            text-decoration: none;
            box-shadow: 0 5px 15px rgba(139, 0, 0, 0.4);
            transition: all 0.3s;
        }
        
        .admin-link:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(139, 0, 0, 0.6);
            color: white;
        }
    </style>
</head>
<body>
    <!-- Hero Section -->
    <div class="hero-section">
        <div class="container">
            <h3 class="h5 mb-2">
                <i class="fas fa-satellite-dish me-1"></i>Watch Live TV Online
            </h3>
            <p class="small mb-3">Stream your favorite TV channels for free</p>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="container py-3">
        <?php if (empty($channels)): ?>
            <div class="no-channels">
                <i class="fas fa-tv fa-5x mb-4"></i>
                <h3>No Channels Available</h3>
                <p>Channels will appear here once they are added by the administrator.</p>
                <a href="admin/login.php" class="btn btn-primary">
                    <i class="fas fa-cog me-2"></i>Admin Panel
                </a>
            </div>
        <?php else: ?>
            
            <!-- Main Content Layout -->
            <div class="row">
                <!-- Left Side - Video Player -->
                <div class="col-lg-8">
                    <div class="video-player-container mb-4" id="videoPlayerContainer" style="display: none;">
                        <div class="video-player">
                            <video id="videoPlayer" controls autoplay>
                                <source src="" type="application/x-mpegURL">
                                Your browser does not support the video tag.
                            </video>
                        </div>
                        <div class="text-center mt-3">
                            <button class="btn btn-outline-light" onclick="closePlayer()">
                                <i class="fas fa-times me-2"></i>Close Player
                            </button>
                        </div>
                    </div>
                    
                    <!-- Default Message when no video playing -->
                    <div class="welcome-message" id="welcomeMessage">
                        <div class="text-center py-5">
                            <i class="fas fa-play-circle fa-5x text-white mb-4"></i>
                            <h3 class="text-white">Select a Channel to Start Watching</h3>
                            <p class="text-white-50">Choose from the channels on the right to begin streaming</p>
                        </div>
                    </div>
                </div>
                
                <!-- Right Side - Channels -->
                <div class="col-lg-4">
                    <!-- Category Filters -->
                    <div class="category-filter mb-3">
                        <h6 class="text-white mb-2">
                            <i class="fas fa-filter me-1"></i>Filter by Category
                        </h6>
                        <div class="d-flex flex-wrap">
                            <button class="filter-btn active" data-category="all">
                                <i class="fas fa-globe me-1"></i>All
                            </button>
                            <?php foreach ($categories as $category): ?>
                                <button class="filter-btn" data-category="<?php echo htmlspecialchars($category); ?>">
                                    <?php echo htmlspecialchars($category); ?>
                                </button>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    
                    <!-- Channels Grid -->
                    <div id="channelsGrid" class="row">
                        <?php foreach ($channels as $channel): ?>
                            <div class="col-6 channel-item mb-3" 
                                 data-category="<?php echo htmlspecialchars($channel['category'] ?? 'General'); ?>"
                                 data-name="<?php echo strtolower(htmlspecialchars($channel['name'])); ?>">
                                <div class="channel-card <?php echo (isset($channel['is_pinned']) && $channel['is_pinned']) ? 'pinned-channel' : ''; ?>" 
                                     onclick="playChannel('<?php echo htmlspecialchars($channel['stream_url']); ?>', '<?php echo htmlspecialchars($channel['name']); ?>')">
                                    <div class="d-flex align-items-center">
                                        <div class="channel-logo-small me-3">
                                            <?php if ($channel['logo_url']): ?>
                                                <img src="<?php echo htmlspecialchars($channel['logo_url']); ?>" 
                                                     alt="<?php echo htmlspecialchars($channel['name']); ?>" 
                                                     class="channel-logo">
                                            <?php else: ?>
                                                <div class="channel-logo d-flex align-items-center justify-content-center bg-primary text-white">
                                                    <i class="fas fa-tv"></i>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="flex-grow-1">
                                            <h6 class="channel-name mb-1">
                                                <?php if (isset($channel['is_pinned']) && $channel['is_pinned']): ?>
                                                    <i class="fas fa-thumbtack text-warning me-1" title="Pinned Channel"></i>
                                                <?php endif; ?>
                                                <?php echo htmlspecialchars($channel['name']); ?>
                                            </h6>
                                            <span class="channel-category">
                                                <?php echo htmlspecialchars($channel['category'] ?? 'General'); ?>
                                            </span>
                                        </div>
                                        <div class="play-icon">
                                            <i class="fas fa-play-circle fa-lg text-white"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
    
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/hls.js@latest"></script>
    <script>
        
        // Category filter functionality
        document.querySelectorAll('.filter-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                // Remove active class from all buttons
                document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
                // Add active class to clicked button
                this.classList.add('active');
                
                const category = this.getAttribute('data-category');
                const channels = document.querySelectorAll('.channel-item');
                
                channels.forEach(channel => {
                    if (category === 'all' || channel.getAttribute('data-category') === category) {
                        channel.style.display = 'block';
                    } else {
                        channel.style.display = 'none';
                    }
                });
            });
        });
        
        // Video player functionality
        function playChannel(streamUrl, channelName) {
            const videoPlayer = document.getElementById('videoPlayer');
            const playerContainer = document.getElementById('videoPlayerContainer');
            const welcomeMessage = document.getElementById('welcomeMessage');
            
            // Show player and hide welcome message
            playerContainer.style.display = 'block';
            welcomeMessage.style.display = 'none';
            
            // Scroll to top to keep player visible
            window.scrollTo({ top: 0, behavior: 'smooth' });
            
            // Set video source
            videoPlayer.src = streamUrl;
            
            // Update page title
            document.title = `Now Playing: ${channelName} - Live TV Stream`;
            
            // Check if HLS is supported
            if (Hls.isSupported()) {
                const hls = new Hls();
                hls.loadSource(streamUrl);
                hls.attachMedia(videoPlayer);
            } else if (videoPlayer.canPlayType('application/vnd.apple.mpegurl')) {
                // Native HLS support (Safari)
                videoPlayer.src = streamUrl;
            } else {
                alert('Your browser does not support HLS streaming. Please try a different browser.');
            }
        }
        
        function closePlayer() {
            const videoPlayer = document.getElementById('videoPlayer');
            const playerContainer = document.getElementById('videoPlayerContainer');
            const welcomeMessage = document.getElementById('welcomeMessage');
            
            // Stop video
            videoPlayer.pause();
            videoPlayer.src = '';
            
            // Hide player and show welcome message
            playerContainer.style.display = 'none';
            welcomeMessage.style.display = 'block';
            
            // Reset page title
            document.title = 'Live TV Streaming - Watch Free TV Online';
        }
        
        // Auto-hide player on page load
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('videoPlayerContainer').style.display = 'none';
        });
    </script>
</body>
</html>
