<?php
require_once '../config/config.php';
require_once '../includes/m3u_parser.php';
requireLogin();

$pdo = getDBConnection();
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'upload_file') {
        // Handle M3U file upload
        if (isset($_FILES['m3u_file']) && $_FILES['m3u_file']['error'] === UPLOAD_ERR_OK) {
            $uploadResult = uploadFile($_FILES['m3u_file'], M3U_UPLOAD_PATH, ['m3u', 'txt']);
            
            if ($uploadResult['success']) {
                $m3uPath = $uploadResult['file_path'];
                $channels = M3UParser::parseM3UFromFile($m3uPath);
                
                if ($channels) {
                    $addedCount = 0;
                    foreach ($channels as $channel) {
                        try {
                            $stmt = $pdo->prepare("INSERT INTO tv_channels (name, stream_url, logo_url, category) VALUES (?, ?, ?, ?)");
                            $stmt->execute([
                                $channel['name'],
                                $channel['stream_url'],
                                $channel['logo_url'],
                                $channel['category']
                            ]);
                            $addedCount++;
                        } catch (Exception $e) {
                            // Skip duplicate or invalid entries
                            continue;
                        }
                    }
                    $message = "Successfully added {$addedCount} channels from M3U file.";
                } else {
                    $error = "No valid channels found in the M3U file.";
                }
            } else {
                $error = $uploadResult['message'];
            }
        } else {
            $error = "Please select a valid M3U file.";
        }
    }
    
    elseif ($action === 'add_single') {
        // Handle single channel addition
        $name = sanitizeInput($_POST['channel_name']);
        $streamUrl = sanitizeInput($_POST['stream_url']);
        $category = sanitizeInput($_POST['category']);
        $logoUrl = sanitizeInput($_POST['logo_url']);
        $logoPath = '';
        
        // Handle logo upload
        if (isset($_FILES['logo_file']) && $_FILES['logo_file']['error'] === UPLOAD_ERR_OK) {
            $logoUpload = uploadFile($_FILES['logo_file'], LOGO_UPLOAD_PATH, ['jpg', 'jpeg', 'png', 'gif', 'webp']);
            if ($logoUpload['success']) {
                $logoPath = $logoUpload['file_path'];
            }
        }
        
        // Use logo URL if no file uploaded
        if (empty($logoPath) && !empty($logoUrl)) {
            $logoPath = $logoUrl;
        }
        
        if (!empty($name) && !empty($streamUrl)) {
            try {
                $stmt = $pdo->prepare("INSERT INTO tv_channels (name, stream_url, category, logo_url) VALUES (?, ?, ?, ?)");
                $stmt->execute([$name, $streamUrl, $category, $logoPath]);
                $message = "Channel '{$name}' added successfully.";
            } catch (Exception $e) {
                $error = "Error adding channel: " . $e->getMessage();
            }
        } else {
            $error = "Please fill in all required fields.";
        }
    }
    
    elseif ($action === 'add_from_url') {
        // Handle M3U URL
        $m3uUrl = sanitizeInput($_POST['m3u_url']);
        
        if (!empty($m3uUrl) && filter_var($m3uUrl, FILTER_VALIDATE_URL)) {
            $channels = M3UParser::parseM3UFromUrl($m3uUrl);
            
            if ($channels) {
                $addedCount = 0;
                foreach ($channels as $channel) {
                    try {
                        $stmt = $pdo->prepare("INSERT INTO tv_channels (name, stream_url, logo_url, category) VALUES (?, ?, ?, ?)");
                        $stmt->execute([
                            $channel['name'],
                            $channel['stream_url'],
                            $channel['logo_url'],
                            $channel['category']
                        ]);
                        $addedCount++;
                    } catch (Exception $e) {
                        continue;
                    }
                }
                $message = "Successfully added {$addedCount} channels from M3U URL.";
            } else {
                $error = "No valid channels found in the M3U URL.";
            }
        } else {
            $error = "Please enter a valid M3U URL.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload M3U - Live TV Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            min-height: 100vh;
            color: white;
        }
        .sidebar .nav-link {
            color: rgba(255, 255, 255, 0.8);
            padding: 12px 20px;
            border-radius: 8px;
            margin: 5px 10px;
            transition: all 0.3s;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.2);
            transform: translateX(5px);
        }
        .main-content {
            background: #000000;
            min-height: 100vh;
            color: white;
        }
        
        .main-content h1, .main-content h2, .main-content h3, .main-content h4, .main-content h5, .main-content h6 {
            color: white !important;
        }
        
        .main-content p, .main-content span, .main-content div {
            color: white !important;
        }
        
        .main-content .card-header {
            background: rgba(220, 20, 60, 0.2);
            color: white;
            border-bottom: 1px solid rgba(220, 20, 60, 0.3);
        }
        
        .main-content .card-body {
            background: rgba(255, 255, 255, 0.05);
            color: white;
        }
        
        .main-content .form-label {
            color: white !important;
        }
        
        .main-content .form-control {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
        }
        
        .main-content .form-control::placeholder {
            color: rgba(255, 255, 255, 0.7);
        }
        
        .main-content .form-select {
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
        }
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
        }
        .btn-primary {
            background: linear-gradient(135deg, #8B0000 0%, #DC143C 100%);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            font-weight: 600;
        }
        .upload-area {
            border: 2px dashed #DC143C;
            border-radius: 15px;
            padding: 3rem;
            text-align: center;
            background: rgba(220, 20, 60, 0.05);
            transition: all 0.3s;
        }
        .upload-area:hover {
            background: rgba(220, 20, 60, 0.1);
            border-color: #B22222;
        }
        .upload-area.dragover {
            background: rgba(220, 20, 60, 0.2);
            border-color: #8B0000;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0">
                <div class="sidebar">
                    <div class="p-3">
                        <h4><i class="fas fa-tv me-2"></i>Live TV Admin</h4>
                        <p class="text-light">Welcome, <?php echo $_SESSION['admin_username']; ?></p>
                    </div>
                    <nav class="nav flex-column">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="channels.php">
                            <i class="fas fa-tv me-2"></i>Manage Channels
                        </a>
                        <a class="nav-link active" href="upload-m3u.php">
                            <i class="fas fa-upload me-2"></i>Upload M3U
                        </a>
                        <a class="nav-link" href="playlists.php">
                            <i class="fas fa-list me-2"></i>M3U Playlists
                        </a>
                        <a class="nav-link" href="../index.php" target="_blank">
                            <i class="fas fa-external-link-alt me-2"></i>View Website
                        </a>
                        <a class="nav-link" href="logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Logout
                        </a>
                    </nav>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="main-content p-4">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2><i class="fas fa-upload me-2"></i>Add TV Channels</h2>
                        <a href="channels.php" class="btn btn-outline-primary">
                            <i class="fas fa-list me-2"></i>View All Channels
                        </a>
                    </div>
                    
                    <?php if ($message): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <i class="fas fa-check-circle me-2"></i><?php echo $message; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Upload Methods -->
                    <div class="row">
                        <!-- Upload M3U File -->
                        <div class="col-md-6 mb-4">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0"><i class="fas fa-file-upload me-2"></i>Upload M3U File</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST" enctype="multipart/form-data">
                                        <input type="hidden" name="action" value="upload_file">
                                        
                                        <div class="upload-area" id="uploadArea">
                                            <i class="fas fa-cloud-upload-alt fa-3x text-primary mb-3"></i>
                                            <h5>Drop M3U file here or click to browse</h5>
                                            <p class="text-muted">Supported formats: .m3u, .txt</p>
                                            <input type="file" name="m3u_file" id="m3uFile" accept=".m3u,.txt" class="d-none" required>
                                            <button type="button" class="btn btn-outline-primary" onclick="document.getElementById('m3uFile').click()">
                                                <i class="fas fa-folder-open me-2"></i>Choose File
                                            </button>
                                        </div>
                                        
                                        <div class="mt-3">
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="fas fa-upload me-2"></i>Upload & Parse M3U
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Add Single Channel -->
                        <div class="col-md-6 mb-4">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0"><i class="fas fa-plus me-2"></i>Add Single Channel</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST" enctype="multipart/form-data">
                                        <input type="hidden" name="action" value="add_single">
                                        
                                        <div class="mb-3">
                                            <label for="channel_name" class="form-label">Channel Name</label>
                                            <input type="text" class="form-control" id="channel_name" name="channel_name" required>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="stream_url" class="form-label">Stream URL</label>
                                            <input type="url" class="form-control" id="stream_url" name="stream_url" 
                                                   placeholder="http://example.com/stream.m3u8" required>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="category" class="form-label">Category</label>
                                            <select class="form-select" id="category" name="category">
                                                <option value="General">General</option>
                                                <option value="News">News</option>
                                                <option value="Sports">Sports</option>
                                                <option value="Movies">Movies</option>
                                                <option value="Kids">Kids</option>
                                                <option value="Music">Music</option>
                                                <option value="Documentary">Documentary</option>
                                            </select>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="logo_url" class="form-label">Logo URL (Optional)</label>
                                            <input type="url" class="form-control" id="logo_url" name="logo_url" 
                                                   placeholder="https://example.com/logo.png">
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="logo_file" class="form-label">Or Upload Logo File</label>
                                            <input type="file" class="form-control" id="logo_file" name="logo_file" 
                                                   accept="image/*">
                                        </div>
                                        
                                        <button type="submit" class="btn btn-primary w-100">
                                            <i class="fas fa-plus me-2"></i>Add Channel
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Add from M3U URL -->
                    <div class="row">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="mb-0"><i class="fas fa-link me-2"></i>Add from M3U URL</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <input type="hidden" name="action" value="add_from_url">
                                        
                                        <div class="row">
                                            <div class="col-md-8">
                                                <label for="m3u_url" class="form-label">M3U Playlist URL</label>
                                                <input type="url" class="form-control" id="m3u_url" name="m3u_url" 
                                                       placeholder="https://example.com/playlist.m3u" required>
                                            </div>
                                            <div class="col-md-4">
                                                <label class="form-label">&nbsp;</label>
                                                <button type="submit" class="btn btn-primary w-100">
                                                    <i class="fas fa-download me-2"></i>Import from URL
                                                </button>
                                            </div>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // File upload drag and drop
        const uploadArea = document.getElementById('uploadArea');
        const fileInput = document.getElementById('m3uFile');
        
        uploadArea.addEventListener('click', () => fileInput.click());
        
        uploadArea.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadArea.classList.add('dragover');
        });
        
        uploadArea.addEventListener('dragleave', () => {
            uploadArea.classList.remove('dragover');
        });
        
        uploadArea.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadArea.classList.remove('dragover');
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                fileInput.files = files;
                updateFileDisplay();
            }
        });
        
        fileInput.addEventListener('change', updateFileDisplay);
        
        function updateFileDisplay() {
            if (fileInput.files.length > 0) {
                const fileName = fileInput.files[0].name;
                uploadArea.innerHTML = `
                    <i class="fas fa-file-alt fa-3x text-success mb-3"></i>
                    <h5>Selected: ${fileName}</h5>
                    <p class="text-muted">Ready to upload</p>
                    <button type="button" class="btn btn-outline-secondary" onclick="resetUpload()">
                        <i class="fas fa-times me-2"></i>Change File
                    </button>
                `;
            }
        }
        
        function resetUpload() {
            fileInput.value = '';
            uploadArea.innerHTML = `
                <i class="fas fa-cloud-upload-alt fa-3x text-primary mb-3"></i>
                <h5>Drop M3U file here or click to browse</h5>
                <p class="text-muted">Supported formats: .m3u, .txt</p>
                <button type="button" class="btn btn-outline-primary" onclick="document.getElementById('m3uFile').click()">
                    <i class="fas fa-folder-open me-2"></i>Choose File
                </button>
            `;
        }
    </script>
</body>
</html>
